<?php
session_start();
header('Content-Type: application/json');

// Desabilitar exibição de erros no output para não quebrar o JSON
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Verifica sessão
if (!isset($_SESSION['loggedin']) || $_SESSION['loggedin'] !== true) {
    http_response_code(403);
    echo json_encode(['error' => 'Acesso não autorizado']);
    exit;
}

require_once 'db_config.php';

$doctor_id = $_SESSION['doctor_id'];
$canal = $_GET['canal'] ?? '';
$status = $_GET['status'] ?? '';

// Parâmetros de Datas
$startA = $_GET['startA'] ?? '';
$endA = $_GET['endA'] ?? '';
$startB = $_GET['startB'] ?? '';
$endB = $_GET['endB'] ?? '';

function getPeriodData($conn, $doctor_id, $startDate, $endDate, $canal, $status) {
    
    // --- FILTROS DE DATA (Trafego e Organico usam coluna 'data') ---
    $whereClauseData = " WHERE doctor_id = ?";
    $paramsData = [$doctor_id];
    $typesData = 'i';

    if (!empty($startDate) && !empty($endDate)) {
        $whereClauseData .= " AND data BETWEEN ? AND ?";
        $paramsData[] = $startDate;
        $paramsData[] = $endDate;
        $typesData .= 'ss';
    }

    // --- FILTROS DE LEADS (Leads usam coluna 'data_insercao' ou 'data') ---
    // Nota: Para compatibilidade com os gráficos do dashboard, vamos garantir que temos acesso à data
    $whereClauseLeads = " WHERE doctor_id = ?";
    $paramsLeads = [$doctor_id];
    $typesLeads = 'i';

    if (!empty($startDate) && !empty($endDate)) {
        // Assumindo que no resumo você usa data_insercao
        $whereClauseLeads .= " AND data_insercao BETWEEN ? AND ?";
        $paramsLeads[] = $startDate;
        $paramsLeads[] = $endDate;
        $typesLeads .= 'ss';
    }
    if (!empty($canal)) {
        $whereClauseLeads .= " AND canal_de_origem = ?";
        $paramsLeads[] = $canal;
        $typesLeads .= 's';
    }
    if (!empty($status)) {
        $whereClauseLeads .= " AND status = ?";
        $paramsLeads[] = $status;
        $typesLeads .= 's';
    }

    // --- 1. DADOS BRUTOS (NOVO - PARA OS GRÁFICOS DE LINHA DO TEMPO) ---
    // Precisamos buscar as listas para montar os gráficos de dia a dia no JS
    
    // Leads: Selecionamos 'data_insercao' as 'data' para o JS entender igual ao dashboard central
    $sqlRawLeads = "SELECT *, data_insercao as data FROM leads {$whereClauseLeads} ORDER BY data_insercao DESC";
    $rawLeads = fetchData($conn, $sqlRawLeads, $typesLeads, $paramsLeads);

    // Tráfego Pago:
    $sqlRawTrafego = "SELECT * FROM trafegoPago {$whereClauseData} ORDER BY data DESC";
    $rawTrafego = fetchData($conn, $sqlRawTrafego, $typesData, $paramsData);

    // Orgânico:
    $sqlRawOrganico = "SELECT * FROM organico {$whereClauseData} ORDER BY data DESC";
    $rawOrganico = fetchData($conn, $sqlRawOrganico, $typesData, $paramsData);


    // --- 2. QUERY PRINCIPAL (FUNIL - MANTIDO) ---
    $sqlFunnel = "
        SELECT
        (SELECT COALESCE(SUM(impressoes), 0) FROM trafegoPago {$whereClauseData}) as impressoes,
        (
            (SELECT COALESCE(SUM(cliques_totais), 0) FROM trafegoPago {$whereClauseData}) + 
            (SELECT COALESCE(SUM(cliques_no_link), 0) FROM organico {$whereClauseData}) +
            (SELECT COALESCE(SUM(visitas_no_perfil), 0) FROM organico {$whereClauseData})
        ) as cliques,
        (SELECT COALESCE(COUNT(id), 0) FROM leads {$whereClauseLeads}) as leads,
        (SELECT COALESCE(COUNT(id), 0) FROM leads {$whereClauseLeads} AND (status = 'Em Contato' OR status LIKE '%Agenda%' OR status LIKE '%Marcad%')) as agendados,
        (SELECT COALESCE(COUNT(id), 0) FROM leads {$whereClauseLeads} AND status = 'Realizado') as realizadas,
        (SELECT COALESCE(SUM(investimento_total), 0) FROM trafegoPago {$whereClauseData}) as investimento_total
    ";
    
    $funnel_params = array_merge($paramsData, $paramsData, $paramsData, $paramsData, $paramsLeads, $paramsLeads, $paramsLeads, $paramsData);
    $funnel_types = $typesData . $typesData . $typesData . $typesData . $typesLeads . $typesLeads . $typesLeads . $typesData;
    
    $funnelResult = fetchData($conn, $sqlFunnel, $funnel_types, $funnel_params);
    $funnelData = $funnelResult[0] ?? ['impressoes'=>0, 'cliques'=>0, 'leads'=>0, 'agendados'=>0, 'realizadas'=>0, 'investimento_total'=>0];

    // --- 3. KPIs Financeiros (Auxiliares - MANTIDO) ---
    $sqlKpis = "
        SELECT
        (SELECT COALESCE(SUM(leads), 0) FROM trafegoPago {$whereClauseData}) as total_paid_leads,
        (SELECT COALESCE(SUM(novos_seguidores), 0) FROM organico {$whereClauseData}) as new_followers
    ";
    $kpi_params = array_merge($paramsData, $paramsData);
    $kpi_types = $typesData . $typesData;
    
    $kpiResult = fetchData($conn, $sqlKpis, $kpi_types, $kpi_params);
    $kpiData = $kpiResult[0] ?? ['total_paid_leads'=>0, 'new_followers'=>0];

    // --- 4. Gráficos Originais do Resumo (MANTIDO) ---
    $sqlMotive = "SELECT motivo, COUNT(id) as total FROM leads {$whereClauseLeads} AND motivo IS NOT NULL AND motivo != '' GROUP BY motivo ORDER BY total DESC LIMIT 7";
    
    // Simplificando Performance (Pie Chart original do resumo)
    $sqlPerformance = "
        SELECT data_insercao as data, 
        SUM(CASE WHEN canal_de_origem = 'Instagram' OR canal_de_origem = 'Facebook' THEN 1 ELSE 0 END) as organic_leads,
        SUM(CASE WHEN canal_de_origem LIKE '%Anuncio%' OR canal_de_origem LIKE '%Ads%' THEN 1 ELSE 0 END) as paid_leads
        FROM leads {$whereClauseLeads}
        GROUP BY data_insercao
        ORDER BY data_insercao ASC
    ";

    // Dados de Perfil
    $profileDataRow = ['name' => 'Doutor', 'url' => null];
    $sqlProfile = "SELECT name, foto_url AS url FROM doctors WHERE id = ?";
    $pRes = fetchData($conn, $sqlProfile, 'i', [$doctor_id]);
    if($pRes) $profileDataRow = $pRes[0];

    // Filtros Dropdown
    $canais = fetchData($conn, "SELECT DISTINCT canal_de_origem FROM leads WHERE doctor_id = ? AND canal_de_origem != ''", 'i', [$doctor_id]);
    $statuses = fetchData($conn, "SELECT DISTINCT status FROM leads WHERE doctor_id = ? AND status != ''", 'i', [$doctor_id]);

    return [
        'profile' => $profileDataRow,
        'filterOptions' => [
            'canais' => $canais,
            'status' => $statuses
        ],
        // MANTIDO TUDO IGUAL
        'funnel' => $funnelData,
        'kpis' => $kpiData,
        'charts' => [
            'leadMotive' => fetchData($conn, $sqlMotive, $typesLeads, $paramsLeads),
            'performance' => fetchData($conn, $sqlPerformance, $typesLeads, $paramsLeads)
        ],
        // NOVO: Adicionamos os dados crus para os gráficos avançados
        'raw_data' => [
            'leads' => $rawLeads,
            'trafego' => $rawTrafego,
            'organico' => $rawOrganico
        ]
    ];
}

function fetchData($conn, $sql, $types, $params) {
    $stmt = $conn->prepare($sql);
    if (!$stmt) return false;
    
    if (!empty($types) && !empty($params)) {
        $stmt->bind_param($types, ...$params);
    }
    
    if (!$stmt->execute()) return false;
    
    $result = $stmt->get_result();
    $data = [];
    while ($row = $result->fetch_assoc()) {
        $data[] = $row;
    }
    $stmt->close();
    return $data;
}

// Execução
try {
    $dataA = getPeriodData($conn, $doctor_id, $startA, $endA, $canal, $status);

    $dataB = null;
    if (!empty($startB) && !empty($endB)) {
        // No período B, não precisamos necessariamente dos raw_data para gráficos pesados,
        // pois geralmente só comparamos KPIs, mas vamos manter a estrutura.
        $dataB = getPeriodData($conn, $doctor_id, $startB, $endB, $canal, $status);
    }

    echo json_encode([
        'current' => $dataA,
        'previous' => $dataB
    ]);
} catch (Exception $e) {
    echo json_encode(['error' => $e->getMessage()]);
}

$conn->close();
?>