// =================================================================
// PAINEL ZORDON - SCRIPT DE ALTA PERFORMANCE v4.1 (VERSÃO CORRIGIDA)
// Biblioteca: Google Charts
// Foco: Correção de performance com filtragem de dados no back-end.
// =================================================================

google.charts.load('current', { 'packages': ['corechart'] });
google.charts.setOnLoadCallback(run);

const state = {
    allLeads: [],
    allMetrics: [],
    filteredLeads: [],
    // A métrica filtrada não é mais necessária, pois o back-end já faz isso.
};

async function run() {
    const elements = {
        doctorName: document.getElementById('doctorName'),
        logoutButton: document.getElementById('logoutButton'),
        dateRangeSelect: document.getElementById('dateRange'),
        customDateFilters: document.getElementById('customDateFilters'),
        startDateInput: document.getElementById('startDate'),
        endDateInput: document.getElementById('endDate'),
        statusFilter: document.getElementById('statusFilter'),
        channelFilter: document.getElementById('channelFilter'),
        applyFiltersButton: document.getElementById('applyFiltersButton')
    };

    // --- LÓGICA DE DADOS ---

    const checkSession = async () => {
        try {
            const response = await fetch('api/auth.php?action=check_session');
            if (!response.ok) throw new Error('Falha na rede');
            const result = await response.json();
            if (!result.loggedin) {
                window.location.href = 'login.html';
                return false;
            }
            elements.doctorName.textContent = `Olá, ${result.doctor_name}`;
            return true;
        } catch (e) {
            window.location.href = 'login.html';
            return false;
        }
    };

    // --- MUDANÇA CRÍTICA: Busca de dados agora envia os filtros de data ---
    const fetchData = async () => {
        const { startDate, endDate } = getDateRange();
        // Formata as datas para YYYY-MM-DD ou string vazia se forem nulas
        const startDateString = startDate ? startDate.toISOString().split('T')[0] : '';
        const endDateString = endDate ? endDate.toISOString().split('T')[0] : '';

        const baseUrl = 'api/data_handler.php';
        const params = `&startDate=${startDateString}&endDate=${endDateString}`;

        try {
            const [metricsRes, leadsRes] = await Promise.all([
                fetch(`${baseUrl}?type=metrics${params}`),
                fetch(`${baseUrl}?type=leads${params}`)
            ]);
            if (!metricsRes.ok || !leadsRes.ok) throw new Error('Falha ao buscar dados da API');

            // Os dados recebidos já são pré-filtrados por data
            state.allMetrics = await metricsRes.json();
            state.allLeads = await leadsRes.json();
            
        } catch (error) {
            console.error("Erro fatal ao buscar dados:", error);
        }
    };

    // Popula os filtros de Status e Canal uma única vez no início
    const populateStaticFilters = async () => {
        // Para popular os filtros com todas as opções, fazemos uma busca inicial sem data
        const baseUrl = 'api/data_handler.php';
        try {
            const leadsRes = await fetch(`${baseUrl}?type=leads`);
            const allLeadsForFilters = await leadsRes.json();

            const statuses = [...new Set(allLeadsForFilters.map(lead => lead.status).filter(Boolean))];
            const channels = [...new Set(allLeadsForFilters.map(lead => lead.canal_de_origem).filter(Boolean))];

            elements.statusFilter.innerHTML = '<option value="all">Todos os Status</option>';
            statuses.forEach(status => elements.statusFilter.innerHTML += `<option value="${status}">${status}</option>`);

            elements.channelFilter.innerHTML = '<option value="all">Todos os Canais</option>';
            channels.forEach(channel => elements.channelFilter.innerHTML += `<option value="${channel}">${channel}</option>`);
        } catch (error) {
            console.error("Erro ao popular filtros:", error);
        }
    };

    const getDateRange = () => {
        const range = elements.dateRangeSelect.value;
        const today = new Date();
        today.setHours(0, 0, 0, 0);
        let startDate, endDate = new Date();
        endDate.setHours(23, 59, 59, 999);

        switch (range) {
            case 'today':
                startDate = new Date(today);
                break;
            case 'this_week':
                startDate = new Date(today);
                startDate.setDate(startDate.getDate() - today.getDay());
                break;
            case 'this_month':
                startDate = new Date(today.getFullYear(), today.getMonth(), 1);
                break;
            case 'last_30_days':
                startDate = new Date(today);
                startDate.setDate(startDate.getDate() - 30);
                break;
            case 'custom':
                startDate = elements.startDateInput.value ? new Date(elements.startDateInput.value + "T00:00:00") : null;
                endDate = elements.endDateInput.value ? new Date(elements.endDateInput.value + "T23:59:59") : null;
                break;
            default: // 'all'
                startDate = null;
                endDate = null;
                break;
        }
        return { startDate, endDate };
    };
    
    // --- MUDANÇA DE LÓGICA: Esta função agora filtra apenas no front-end o que já foi pré-filtrado (status/canal) ---
    const applyClientSideFilters = () => {
        state.filteredLeads = state.allLeads.filter(lead => {
            const statusMatch = elements.statusFilter.value === 'all' || lead.status === elements.statusFilter.value;
            const channelMatch = elements.channelFilter.value === 'all' || lead.canal_de_origem === elements.channelFilter.value;
            return statusMatch && channelMatch;
        });
    };

    // --- FUNÇÕES DE RENDERIZAÇÃO (praticamente sem alterações) ---

    const render = () => {
        renderKpiCards();
        renderLeadsTable();
        renderAllCharts();
    };

    const renderKpiCards = () => {
        // Usa `state.filteredLeads` para leads (que considera status/canal) 
        // e `state.allMetrics` para métricas (que considera apenas o período)
        const leads = state.filteredLeads;
        const metrics = state.allMetrics;
        
        const totalInvestment = metrics.reduce((sum, item) => sum + parseFloat(item.investimento_total || 0), 0);
        const agendados = leads.filter(l => l.status === 'Agendado');
        const totalRevenue = agendados.reduce((sum, item) => sum + parseFloat(item.preco || 0), 0);
        const roi = totalInvestment > 0 ? (totalRevenue / totalInvestment) : 0;
        const totalLeads = leads.length; // Total de leads *após* filtros de status/canal
        const cpl = totalLeads > 0 ? (totalInvestment / totalLeads) : 0;
        
        // A taxa de conversão deve considerar todos os leads do período, antes do filtro de status/canal
        const conversionRate = state.allLeads.length > 0 ? (agendados.length / state.allLeads.length) * 100 : 0;

        document.getElementById('totalInvestmentCard').textContent = `R$ ${totalInvestment.toLocaleString('pt-BR', { minimumFractionDigits: 2, maximumFractionDigits: 2 })}`;
        document.getElementById('totalRevenueCard').textContent = `R$ ${totalRevenue.toLocaleString('pt-BR', { minimumFractionDigits: 2, maximumFractionDigits: 2 })}`;
        document.getElementById('roiCard').textContent = `${roi.toFixed(2)}x`;
        document.getElementById('totalLeadsCard').textContent = totalLeads;
        document.getElementById('cplCard').textContent = `R$ ${cpl.toLocaleString('pt-BR', { minimumFractionDigits: 2, maximumFractionDigits: 2 })}`;
        document.getElementById('conversionRateCard').textContent = `${conversionRate.toFixed(1)}%`;
    };

    const renderLeadsTable = () => {
        const tableBody = document.querySelector('#leadsTable tbody');
        tableBody.innerHTML = state.filteredLeads.length === 0
            ? '<tr><td colspan="6">Nenhum lead encontrado para os filtros selecionados.</td></tr>'
            : state.filteredLeads.map(lead => `
                <tr>
                    <td>${new Date(lead.data + 'T00:00:00').toLocaleDateString('pt-BR')}</td>
                    <td>${lead.contato || 'N/A'}</td>
                    <td>${lead.telefone || 'N/A'}</td>
                    <td>${lead.canal_de_origem || 'N/A'}</td>
                    <td>${lead.status || 'N/A'}</td>
                    <td>${parseFloat(lead.preco || 0).toLocaleString('pt-BR', { style: 'currency', currency: 'BRL' })}</td>
                </tr>
            `).join('');
    };

    const renderAllCharts = () => {
        drawPieChart('statusChart', 'Leads por Status', state.filteredLeads, 'status');
        drawPieChart('channelChart', 'Leads por Canal', state.filteredLeads, 'canal_de_origem');
        drawComboChart('investmentChart', 'Performance por Dia', state.allLeads, state.allMetrics);
    };
    
    // As funções dos gráficos (drawPieChart, drawComboChart) não precisam de alteração.
    const drawPieChart = (elementId, title, data, groupBy) => {
        const container = document.getElementById(elementId);
        if (!container) return;
        const groupedData = data.reduce((acc, item) => {
            const key = item[groupBy] || 'Não definido';
            acc[key] = (acc[key] || 0) + 1;
            return acc;
        }, {});
        const dataArray = [[groupBy, 'Quantidade']];
        for (const key in groupedData) { dataArray.push([key, groupedData[key]]); }
        if (dataArray.length <= 1) {
            container.innerHTML = `<p style="text-align:center; padding-top: 50px; color: #999;">Dados insuficientes para o gráfico de ${title}.</p>`;
            return;
        }
        const dataTable = google.visualization.arrayToDataTable(dataArray);
        const options = { title: title, pieHole: 0.4, colors: ['#0d6efd', '#6c757d', '#198754', '#ffc107', '#dc3545', '#0dcaf0'], chartArea: {'width': '90%', 'height': '80%'}, legend: {'position': 'bottom'} };
        const chart = new google.visualization.PieChart(container);
        chart.draw(dataTable, options);
    };

    const drawComboChart = (elementId, title, leads, metrics) => {
        const container = document.getElementById(elementId);
        if (!container) return;
        const dataByDate = {};
        metrics.forEach(m => {
            const date = new Date(m.data + 'T00:00:00');
            dataByDate[date] = { ...dataByDate[date], investment: parseFloat(m.investimento_total || 0) };
        });
        leads.forEach(l => {
            const date = new Date(l.data + 'T00:00:00');
            dataByDate[date] = { ...dataByDate[date], leads: (dataByDate[date]?.leads || 0) + 1 };
            if (l.status === 'Agendado') {
                dataByDate[date] = { ...dataByDate[date], revenue: (dataByDate[date]?.revenue || 0) + parseFloat(l.preco || 0) };
            }
        });
        const sortedDates = Object.keys(dataByDate).sort((a,b) => new Date(a) - new Date(b));
        if (sortedDates.length === 0) {
            container.innerHTML = `<p style="text-align:center; padding-top: 50px; color: #999;">Dados insuficientes para o gráfico de ${title}.</p>`;
            return;
        }
        const dataArray = [['Data', 'Investimento', 'Receita', 'Leads']];
        sortedDates.forEach(dateStr => {
            const date = new Date(dateStr);
            const dataPoint = dataByDate[dateStr];
            dataArray.push([ date, dataPoint.investment || 0, dataPoint.revenue || 0, dataPoint.leads || 0 ]);
        });
        const dataTable = google.visualization.arrayToDataTable(dataArray);
        const options = { title: title, vAxes: { 0: {title: 'Valor (R$)'}, 1: {title: 'Quantidade'} }, hAxis: {title: 'Data', format: 'dd/MM' }, seriesType: 'bars', series: { 0: {type: 'line', targetAxisIndex: 0}, 1: {type: 'line', targetAxisIndex: 0}, 2: {type: 'bars', targetAxisIndex: 1} }, colors: ['#ffc107', '#198754', '#0d6efd'], chartArea: {'width': '85%', 'height': '70%'}, legend: {'position': 'top'} };
        const chart = new google.visualization.ComboChart(container);
        chart.draw(dataTable, options);
    };


    // --- EXECUÇÃO INICIAL E EVENTOS ---

    // Nova função principal que controla o fluxo de dados e renderização
    const handleFilterChange = async () => {
        await fetchData(); // 1. Busca os dados do servidor JÁ FILTRADOS por data.
        applyClientSideFilters(); // 2. Aplica filtros de status/canal sobre os dados recebidos.
        render(); // 3. Desenha a tela com os dados finais.
    };

    if (await checkSession()) {
        await populateStaticFilters(); // Popula os filtros de status/canal uma vez.
        await handleFilterChange(); // Busca os dados iniciais e renderiza a tela.

        // Adiciona os eventos aos botões e seletores
        elements.dateRangeSelect.addEventListener('change', () => {
            elements.customDateFilters.style.display = elements.dateRangeSelect.value === 'custom' ? 'flex' : 'none';
        });
        elements.applyFiltersButton.addEventListener('click', handleFilterChange);
        elements.logoutButton.addEventListener('click', async () => {
            await fetch('api/auth.php?action=logout');
            window.location.href = 'login.html';
        });
        
        // Redesenha os gráficos ao redimensionar a janela para manter a responsividade
        window.addEventListener('resize', renderAllCharts);
    }
}