// =========================================================================
// PAINEL CÓRTEX - SCRIPT DE RESUMO GERAL v1.0
// =========================================================================

google.charts.load('current', {'packages':['corechart', 'bar']});
google.charts.setOnLoadCallback(run);

// O estado agora armazena o objeto JSON complexo da nossa API
const state = {
    allData: null 
};

// --- Funções de Formatação ---
function formatCurrency(value) {
    return parseFloat(value || 0).toLocaleString('pt-BR', { style: 'currency', currency: 'BRL' });
}
function formatNumber(value) {
    return parseInt(value || 0).toLocaleString('pt-BR');
}
function formatPercent(value) {
    return (parseFloat(value || 0) * 100).toFixed(2) + '%';
}

async function run() {
    // 1. Elementos da UI
    const elements = {
        doctorName: document.getElementById('doctorName'),
        logoutButton: document.getElementById('logoutButton'),
        dateRangeSelect: document.getElementById('dateRange'),
        customDateFilters: document.getElementById('customDateFilters'),
        startDateInput: document.getElementById('startDate'),
        endDateInput: document.getElementById('endDate'),
        applyFiltersButton: document.getElementById('applyFiltersButton'),
    };

    // 2. Verificação de Sessão (idêntica à sua)
    const checkSession = async () => {
        try {
            const response = await fetch('api/auth.php?action=check_session');
            if (!response.ok) throw new Error('Falha na rede');
            const result = await response.json();
            if (!result.loggedin) {
                window.location.href = 'login.html';
                return false;
            }
            elements.doctorName.textContent = `Olá, ${result.doctor_name}`;
            return true;
        } catch (e) {
            window.location.href = 'login.html';
            return false;
        }
    };

    // 3. Função para pegar o período (idêntica à sua)
    const getDateRange = (forApi = false) => {
        const range = elements.dateRangeSelect.value;
        const today = new Date();
        let startDate, endDate = new Date();
        
        today.setHours(0, 0, 0, 0);
        endDate.setHours(23, 59, 59, 999);

        switch (range) {
            case 'today': startDate = new Date(today); break;
            case 'this_week': startDate = new Date(today); startDate.setDate(startDate.getDate() - today.getDay()); break;
            case 'this_month': startDate = new Date(today.getFullYear(), today.getMonth(), 1); break;
            case 'last_30_days': startDate = new Date(); startDate.setDate(today.getDate() - 29); startDate.setHours(0,0,0,0); break;
            case 'custom':
                startDate = elements.startDateInput.value ? new Date(elements.startDateInput.value + "T00:00:00") : null;
                endDate = elements.endDateInput.value ? new Date(elements.endDateInput.value + "T23:59:59") : null;
                break;
            default: startDate = null; endDate = null; break;
        }

        if (forApi) {
            const formatDate = (date) => date ? date.toISOString().split('T')[0] : '';
            if (range === 'all') return { startDate: '', endDate: '' };
            return { startDate: formatDate(startDate), endDate: formatDate(endDate) };
        }
        
        return { startDate, endDate };
    };

    // 4. Funções de Renderização
    
    // 4.1. Renderizar KPIs
    const renderKpiCards = () => {
        if (!state.allData || !state.allData.kpis) return;
        const kpis = state.allData.kpis;

        document.getElementById('kpiTotalLeads').textContent = formatNumber(kpis.total_leads);
        document.getElementById('kpiRealizedLeads').textContent = formatNumber(kpis.realized_leads);
        document.getElementById('kpiConversionRate').textContent = formatPercent(kpis.conversion_rate);
        document.getElementById('kpiTotalInvestment').textContent = formatCurrency(kpis.total_investment);
        document.getElementById('kpiCPL').textContent = formatCurrency(kpis.cpl);
        document.getElementById('kpiNewFollowers').textContent = formatNumber(kpis.new_followers);
        document.getElementById('kpiTotalEngagement').textContent = formatNumber(kpis.total_engagement);
        document.getElementById('kpiProfileVisits').textContent = formatNumber(kpis.profile_visits);
    };

    // 4.2. Renderizar TODOS os gráficos
    const renderAllCharts = () => {
        if (!state.allData || !state.allData.charts) return;
        const charts = state.allData.charts;

        drawLeadsOverTimeChart('leadsOverTimeChart', charts.leadsOverTime);
        drawLeadSourcePieChart('leadSourcePieChart', charts.leadSource);
        drawPaidPerformanceChart('paidPerformanceChart', charts.paidPerformance);
        drawOrganicGrowthChart('organicGrowthChart', charts.organicGrowth);
    };
    
    // 4.3. Funções de desenho de cada gráfico
    
    const drawLeadsOverTimeChart = (elementId, data) => {
        const container = document.getElementById(elementId);
        if (!container || data.length === 0) {
            container.innerHTML = `<p class="chart-placeholder">Sem dados de Leads no período.</p>`; return;
        }
        const dataArray = [['Data', 'Leads']];
        data.forEach(item => {
            dataArray.push([new Date(item.data + 'T00:00:00'), parseInt(item.total_leads)]);
        });
        const dataTable = google.visualization.arrayToDataTable(dataArray);
        const options = { title: 'Leads Totais por Dia', hAxis: { format: 'dd/MM' }, legend: { position: 'none' }, series: { 0: { color: '#0d6efd' } } };
        const chart = new google.visualization.LineChart(container);
        chart.draw(dataTable, options);
    };

    const drawLeadSourcePieChart = (elementId, data) => {
        const container = document.getElementById(elementId);
        if (!container || data.length === 0) {
            container.innerHTML = `<p class="chart-placeholder">Sem dados de origem de leads.</p>`; return;
        }
        const dataArray = [['Origem', 'Total']];
        data.forEach(item => {
            dataArray.push([item.canal_de_origem, parseInt(item.total)]);
        });
        const dataTable = google.visualization.arrayToDataTable(dataArray);
        const options = { title: 'Origem dos Leads', is3D: true };
        const chart = new google.visualization.PieChart(container);
        chart.draw(dataTable, options);
    };

    const drawPaidPerformanceChart = (elementId, data) => {
        const container = document.getElementById(elementId);
        if (!container || data.length === 0) {
            container.innerHTML = `<p class="chart-placeholder">Sem dados de Tráfego Pago.</p>`; return;
        }
        const dataArray = [['Data', 'Investimento (R$)', 'Leads']];
        data.forEach(item => {
            dataArray.push([new Date(item.data + 'T00:00:00'), parseFloat(item.investment), parseInt(item.leads)]);
        });
        const dataTable = google.visualization.arrayToDataTable(dataArray);
        const options = {
            title: 'Performance (Pago): Investimento vs. Leads',
            hAxis: { format: 'dd/MM' },
            vAxes: { 0: {title: 'Valor (R$)'}, 1: {title: 'Quantidade'} },
            seriesType: 'bars',
            series: { 0: {type: 'line', targetAxisIndex: 0, color: '#ffc107'}, 1: {type: 'bars', targetAxisIndex: 1, color: '#198754'} }
        };
        const chart = new google.visualization.ComboChart(container);
        chart.draw(dataTable, options);
    };

    const drawOrganicGrowthChart = (elementId, data) => {
        const container = document.getElementById(elementId);
        if (!container || data.length === 0) {
            container.innerHTML = `<p class="chart-placeholder">Sem dados de Crescimento Orgânico.</p>`; return;
        }
        const dataArray = [['Data', 'Novos Seguidores', 'Engajamento']];
        data.forEach(item => {
            dataArray.push([new Date(item.data + 'T00:00:00'), parseInt(item.followers), parseInt(item.engagement)]);
        });
        const dataTable = google.visualization.arrayToDataTable(dataArray);
        const options = {
            title: 'Crescimento Orgânico por Dia',
            hAxis: { format: 'dd/MM' },
            vAxis: { title: 'Quantidade' },
            series: { 0: { color: '#0dcaf0' }, 1: { color: '#d63384' } }
        };
        const chart = new google.visualization.LineChart(container);
        chart.draw(dataTable, options);
    };

    // 5. Função principal para buscar e renderizar TUDO
    const fetchAndRenderAll = async () => {
        try {
            const { startDate, endDate } = getDateRange(true);
            const apiUrl = `api/data_handler_resumo.php?startDate=${startDate}&endDate=${endDate}`;
            
            const response = await fetch(apiUrl);
            if (!response.ok) throw new Error('Falha ao buscar dados do resumo');
            
            const data = await response.json();
            if (data.error) throw new Error(data.error);

            state.allData = data; // Armazena os dados complexos

            // Renderiza tudo
            renderKpiCards();
            renderAllCharts();

        } catch (error) {
            console.error("Erro ao buscar e renderizar dados do resumo:", error);
            // Você pode adicionar uma mensagem de erro na UI aqui
        }
    };

    // 6. Execução e Event Listeners
    if (await checkSession()) {
        await fetchAndRenderAll(); // Carga inicial
        
        // Listener para o botão 'Aplicar' (para datas customizadas)
        elements.applyFiltersButton.addEventListener('click', fetchAndRenderAll);
        
        // Listener para o dropdown de data (para opções não-custom)
        elements.dateRangeSelect.addEventListener('change', async () => {    
            const isCustom = elements.dateRangeSelect.value === 'custom';
            elements.customDateFilters.style.display = isCustom ? 'flex' : 'none';
            if (!isCustom) {
                await fetchAndRenderAll();
            }
        });

        // Listener de Logout
        elements.logoutButton.addEventListener('click', async () => { 
            await fetch('api/auth.php?action=logout'); 
            window.location.href = 'login.html'; 
        });

        // Redesenha os gráficos ao redimensionar a janela
        window.addEventListener('resize', renderAllCharts);
    }
}