// =========================================================================
// PAINEL ZORDON - SCRIPT DE ANÁLISE DE LEADS v3.0 (PAGINAÇÃO CORRIGIDA + FILTROS)
// =========================================================================

google.charts.load('current', {'packages':['corechart']});
google.charts.setOnLoadCallback(run);

const state = {
    allData: [],
    filteredData: []
};

let currentPage = 1;
const rowsPerPage = 15;

async function run() {
    const elements = {
        doctorName: document.getElementById('doctorName'),
        logoutButton: document.getElementById('logoutButton'),
        dateRangeSelect: document.getElementById('dateRange'),
        customDateFilters: document.getElementById('customDateFilters'),
        startDateInput: document.getElementById('startDate'),
        endDateInput: document.getElementById('endDate'),
        statusFilter: document.getElementById('statusFilter'),
        channelFilter: document.getElementById('channelFilter'),
        applyFiltersButton: document.getElementById('applyFiltersButton'),
        contatoFilter: document.getElementById('contatoFilter'),
        paginationControls: document.getElementById('paginationControls')
    };

    const checkSession = async () => {
        try {
            const response = await fetch('api/auth.php?action=check_session');
            if (!response.ok) throw new Error('Falha na rede');
            const result = await response.json();
            if (!result.loggedin) {
                window.location.href = 'login.html';
                return false;
            }
            elements.doctorName.textContent = `Olá, ${result.doctor_name}`;
            return true;
        } catch (e) {
            window.location.href = 'login.html';
            return false;
        }
    };

    const fetchAllData = async () => {
        try {
            const response = await fetch('api/data_handler_leads.php');
            if (!response.ok) throw new Error('Falha ao buscar dados da API de Leads');
            state.allData = await response.json();
        } catch (error) {
            console.error("Erro fatal ao buscar dados de leads:", error);
        }
    };

    const populateFilters = () => {
        const statuses = [...new Set(state.allData.map(lead => lead.status).filter(Boolean))];
        const channels = [...new Set(state.allData.map(lead => lead.canal_de_origem).filter(Boolean))];
        elements.statusFilter.innerHTML = '<option value="all">Todos os Status</option>';
        statuses.forEach(status => {elements.statusFilter.innerHTML += `<option value="${status}">${status}</option>`});
        elements.channelFilter.innerHTML = '<option value="all">Todos os Canais</option>';
        channels.forEach(channel => {elements.channelFilter.innerHTML += `<option value="${channel}">${channel}</option>`});
    };

    const getDateRange = () => {
        const range = elements.dateRangeSelect.value;
        const today = new Date();
        today.setHours(0, 0, 0, 0);
        let startDate, endDate = new Date();
        endDate.setHours(23, 59, 59, 999);
        switch (range) {
            case 'today': startDate = new Date(today); break;
            case 'this_week': startDate = new Date(today); startDate.setDate(startDate.getDate() - today.getDay()); break;
            case 'this_month': startDate = new Date(today.getFullYear(), today.getMonth(), 1); break;
            case 'last_30_days': startDate = new Date(today); startDate.setDate(startDate.getDate() - 30); break;
            case 'custom':
                startDate = elements.startDateInput.value ? new Date(elements.startDateInput.value + "T00:00:00") : null;
                endDate = elements.endDateInput.value ? new Date(elements.endDateInput.value + "T23:59:59") : null;
                break;
            default: startDate = null; endDate = null; break;
        }
        return { startDate, endDate };
    };

    const applyFilters = () => {
        const { startDate, endDate } = getDateRange();
        state.filteredData = state.allData.filter(item => {
            const itemDate = new Date(item.data + 'T00:00:00');
            const statusMatch = elements.statusFilter.value === 'all' || item.status === elements.statusFilter.value;
            const channelMatch = elements.channelFilter.value === 'all' || item.canal_de_origem === elements.channelFilter.value;
            const dateMatch = (!startDate || itemDate >= startDate) && (!endDate || itemDate <= endDate);
            return dateMatch && statusMatch && channelMatch;
        });
    };
    
    const render = () => {
        renderKpiCards();
        renderAllCharts();
    };

    const renderKpiCards = () => {
        const data = state.filteredData;
        const totalLeads = data.length;
        const receitaPotencial = data.reduce((sum, item) => sum + parseFloat(item.preco || 0), 0);
        const leadsAgendados = data.filter(item => item.status === 'Agendado').length;
        const leadsDesqualificados = data.filter(item => item.status === 'Desqualificado').length;
        const precoMedio = totalLeads > 0 ? (receitaPotencial / totalLeads) : 0;
        const taxaAgendamento = totalLeads > 0 ? (leadsAgendados / totalLeads) * 100 : 0;

        document.getElementById('totalLeadsCard').textContent = totalLeads;
        document.getElementById('receitaPotencialCard').textContent = receitaPotencial.toLocaleString('pt-BR', { style: 'currency', currency: 'BRL' });
        document.getElementById('leadsAgendadosCard').textContent = leadsAgendados;
        document.getElementById('leadsDesqualificadosCard').textContent = leadsDesqualificados;
        document.getElementById('precoMedioCard').textContent = precoMedio.toLocaleString('pt-BR', { style: 'currency', currency: 'BRL' });
        document.getElementById('taxaAgendamentoCard').textContent = `${taxaAgendamento.toFixed(1)}%`;
    };

//PAGINAÇÃO

    const populateTableFilters = () => {
        const contatos = [...new Set(state.filteredData.map(item => item.contato).filter(Boolean))];
        contatos.sort();
        elements.contatoFilter.innerHTML = '<option value="all">Todos os Contatos</option>';
        contatos.forEach(contato => {
            elements.contatoFilter.innerHTML += `<option value="${contato}">${contato}</option>`;
        });
    };
    
    const renderTableAndPagination = () => {
        console.log("Renderizando tabela e paginação...");
        const selectedContato = elements.contatoFilter.value;
        const tableFilteredData = state.filteredData.filter(item => {
            return selectedContato === 'all' || item.contato === selectedContato;
        });

        const totalPages = Math.ceil(tableFilteredData.length / rowsPerPage);
        currentPage = Math.min(currentPage, totalPages) || 1;
        const startIndex = (currentPage - 1) * rowsPerPage;
        const paginatedData = tableFilteredData.slice(startIndex, startIndex + rowsPerPage);
        
        console.log(`Página: ${currentPage}, Total de Páginas: ${totalPages}, Itens: ${paginatedData.length}`);

        renderLeadsTable(paginatedData);
        renderPaginationControls(totalPages);
    };
    
    const renderLeadsTable = (data) => {
        const tableBody = document.querySelector('#leadsTable tbody');
        tableBody.innerHTML = data.length === 0 
            ? '<tr><td colspan="6">Nenhum resultado encontrado.</td></tr>'
            : data.map(item => `
                <tr>
                    <td>${new Date(item.data + 'T00:00:00').toLocaleDateString('pt-BR')}</td>
                    <td>${item.contato || 'N/A'}</td>
                    <td>${item.telefone || 'N/A'}</td>
                    <td>${item.canal_de_origem || 'N/A'}</td>
                    <td>${item.status || 'N/A'}</td>
                    <td>${parseFloat(item.preco || 0).toLocaleString('pt-BR', { style: 'currency', currency: 'BRL' })}</td>
                </tr>
            `).join('');
    };

    const renderPaginationControls = (totalPages) => {
        elements.paginationControls.innerHTML = '';
        const prevButton = document.createElement('button');
        prevButton.textContent = '« Anterior';
        prevButton.disabled = currentPage === 1;
        prevButton.addEventListener('click', () => {
            if (currentPage > 1) {
                currentPage--;
                renderTableAndPagination();
            }
        });
        elements.paginationControls.appendChild(prevButton);

        const pageIndicator = document.createElement('span');
        pageIndicator.textContent = `Página ${currentPage} de ${totalPages > 0 ? totalPages : 1}`;
        elements.paginationControls.appendChild(pageIndicator);

        const nextButton = document.createElement('button');
        nextButton.textContent = 'Próximo »';
        nextButton.disabled = currentPage === totalPages || totalPages === 0;
        nextButton.addEventListener('click', () => {
            if (currentPage < totalPages) {
                currentPage++;
                renderTableAndPagination();
            }
        });
        elements.paginationControls.appendChild(nextButton);
    };

    const drawPieChart = (elementId, title, data, groupBy) => {
        const container = document.getElementById(elementId);
        if (!container) return;
        const groupedData = data.reduce((acc, item) => {
            const key = item[groupBy] || 'Não definido';
            acc[key] = (acc[key] || 0) + 1;
            return acc;
        }, {});
        const dataArray = [[groupBy, 'Quantidade']];
        for (const key in groupedData) { dataArray.push([key, groupedData[key]]); }
        if (dataArray.length <= 1) {
            container.innerHTML = `<p style="text-align:center; padding-top: 50px; color: #999;">Dados insuficientes para o gráfico.</p>`;
            return;
        }
        const dataTable = google.visualization.arrayToDataTable(dataArray);
        const options = { title: title, pieHole: 0.4, legend: {'position': 'bottom'} };
        const chart = new google.visualization.PieChart(container);
        chart.draw(dataTable, options);
    };

    const drawMainChart = (elementId, title, data) => {
        const container = document.getElementById(elementId);
        if (!container) return;
        const dataByDate = {};
        data.forEach(item => {
            const date = new Date(item.data + 'T00:00:00').toISOString();
            if (!dataByDate[date]) { dataByDate[date] = { count: 0, revenue: 0 }; }
            dataByDate[date].count += 1;
            dataByDate[date].revenue += parseFloat(item.preco || 0);
        });
        const sortedDates = Object.keys(dataByDate).sort((a,b) => new Date(a) - new Date(b));
        if (sortedDates.length === 0) {
            container.innerHTML = `<p style="text-align:center; padding-top: 50px; color: #999;">Dados insuficientes para o gráfico.</p>`;
            return;
        }
        const dataArray = [['Data', 'Receita Potencial', 'Quantidade de Leads']];
        sortedDates.forEach(dateStr => {
            const date = new Date(dateStr);
            const dataPoint = dataByDate[dateStr];
            dataArray.push([ date, dataPoint.revenue, dataPoint.count ]);
        });
        const dataTable = google.visualization.arrayToDataTable(dataArray);
        const options = { 
            title: title,
            vAxes: { 0: {title: 'Valor (R$)'}, 1: {title: 'Quantidade'} }, 
            hAxis: {title: 'Data', format: 'dd/MM' },
            seriesType: 'bars',
            series: { 0: {type: 'line', targetAxisIndex: 0}, 1: {type: 'bars', targetAxisIndex: 1} }
        };
        const chart = new google.visualization.ComboChart(container);
        chart.draw(dataTable, options);
    };

    const renderAllCharts = () => {
        drawPieChart('statusChart', 'Leads por Status', state.filteredData, 'status');
        drawPieChart('channelChart', 'Leads por Canal de Origem', state.filteredData, 'canal_de_origem');
        drawMainChart('mainChart', 'Leads e Receita Potencial por Dia', state.filteredData);
    };

    const handleFilterChange = () => {
        applyFilters();
        render();
        populateTableFilters();
        currentPage = 1;
        renderTableAndPagination();
    };

    if (await checkSession()) {
        await fetchAllData();
        populateFilters();
        handleFilterChange();

        elements.contatoFilter.addEventListener('change', () => {
            currentPage = 1;
            renderTableAndPagination();
        });
        
        elements.applyFiltersButton.addEventListener('click', handleFilterChange);
        elements.dateRangeSelect.addEventListener('change', () => { elements.customDateFilters.style.display = elements.dateRangeSelect.value === 'custom' ? 'flex' : 'none'; });
        elements.logoutButton.addEventListener('click', async () => { await fetch('api/auth.php?action=logout'); window.location.href = 'login.html'; });
        window.addEventListener('resize', renderAllCharts);
    }
}