// =========================================================================
// PAINEL ZORDON - SCRIPT DE TRÁFEGO PAGO v4.1 (FILTROS CORRIGIDOS)
// =========================================================================

google.charts.load('current', {'packages':['corechart', 'bar']});
google.charts.setOnLoadCallback(run);

const state = {
    allData: [],
    filteredData: []
};

let currentPage = 1;
const rowsPerPage = 15;

function updateTableColumns() {
    const checkboxes = document.querySelectorAll('#columnSelector .dropdown-options input[type="checkbox"]');
    checkboxes.forEach(checkbox => {
        const columnIndex = parseInt(checkbox.getAttribute('data-column')) + 1;
        // AGORA ELE PROCURA PELA TABELA CORRETA!
        const cells = document.querySelectorAll(`#dataTable th:nth-child(${columnIndex}), #dataTable td:nth-child(${columnIndex})`);
        cells.forEach(cell => {
            cell.style.display = checkbox.checked ? '' : 'none';
        });
    });
}

async function run() {
    const elements = {
        doctorName: document.getElementById('doctorName'),
        logoutButton: document.getElementById('logoutButton'),
        dateRangeSelect: document.getElementById('dateRange'),
        customDateFilters: document.getElementById('customDateFilters'),
        startDateInput: document.getElementById('startDate'),
        endDateInput: document.getElementById('endDate'),
        applyFiltersButton: document.getElementById('applyFiltersButton'),
        orderByFilter: document.getElementById('orderByFilter'),
        paginationControls: document.getElementById('paginationControls'),
        minLeadsFilter: document.getElementById('minLeadsFilter'),
        minInvestmentFilter: document.getElementById('minInvestmentFilter'),
        highlightTopResults: document.getElementById('highlightTopResults'),
        columnSelector: document.getElementById('columnSelector'),
        columnSelectorButton: document.querySelector('#columnSelector .dropdown-button'),
        columnCheckboxes: document.querySelectorAll('#columnSelector .dropdown-options input')
    };

    const checkSession = async () => {
        try {
            const response = await fetch('api/auth.php?action=check_session');
            if (!response.ok) throw new Error('Falha na rede');
            const result = await response.json();
            if (!result.loggedin) {
                window.location.href = 'login.html';
                return false;
            }
            elements.doctorName.textContent = `Olá, ${result.doctor_name}`;
            return true;
        } catch (e) {
            window.location.href = 'login.html';
            return false;
        }
    };

    const fetchAllData = async () => {
        try {
            const { startDate, endDate } = getDateRange(true);
            const apiUrl = `api/data_handler_trafego.php?startDate=${startDate}&endDate=${endDate}`;
            const response = await fetch(apiUrl);
            if (!response.ok) throw new Error('Falha ao buscar dados da API de Tráfego');
            
            const data = await response.json();
            if (data.error) throw new Error(data.error);

            state.allData = data;
        } catch (error) {
            console.error("Erro fatal ao buscar dados de tráfego:", error);
        }
    };

    const getDateRange = (forApi = false) => {
        const range = elements.dateRangeSelect.value;
        const today = new Date();
        let startDate, endDate = new Date();
        
        today.setHours(0, 0, 0, 0);
        endDate.setHours(23, 59, 59, 999);

        switch (range) {
            case 'today': startDate = new Date(today); break;
            case 'this_week': startDate = new Date(today); startDate.setDate(startDate.getDate() - today.getDay()); break;
            case 'this_month': startDate = new Date(today.getFullYear(), today.getMonth(), 1); break;
            case 'last_30_days': startDate = new Date(); startDate.setDate(today.getDate() - 29); startDate.setHours(0,0,0,0); break;
            case 'custom':
                startDate = elements.startDateInput.value ? new Date(elements.startDateInput.value + "T00:00:00") : null;
                endDate = elements.endDateInput.value ? new Date(elements.endDateInput.value + "T23:59:59") : null;
                break;
            default: startDate = null; endDate = null; break;
        }

        if (forApi) {
            const formatDate = (date) => date ? date.toISOString().split('T')[0] : '';
            if (range === 'all') return { startDate: '', endDate: '' };
            return { startDate: formatDate(startDate), endDate: formatDate(endDate) };
        }
        
        return { startDate, endDate };
    };
    
    const applyFiltersAndRender = () => {
        const minLeads = parseInt(elements.minLeadsFilter.value) || 0;
        const minInvestment = parseFloat(elements.minInvestmentFilter.value) || 0;
        const orderBy = elements.orderByFilter.value;

        let processedData = state.allData.filter(item => {
            const leads = parseInt(item.leads || 0);
            const investment = parseFloat(item.investimento_total || 0);
            return leads >= minLeads && investment >= minInvestment;
        });

        processedData.sort((a, b) => {
            switch (orderBy) {
                case 'leads_desc': return parseInt(b.leads || 0) - parseInt(a.leads || 0);
                case 'investimento_desc': return parseFloat(b.investimento_total || 0) - parseFloat(a.investimento_total || 0);
                case 'cliques_desc': return parseInt(b.cliques_totais || 0) - parseInt(a.cliques_totais || 0);
                case 'impressoes_desc': return parseInt(b.impressoes || 0) - parseInt(a.impressoes || 0);
                case 'ctr_desc': return parseFloat(b.ctr || 0) - parseFloat(a.ctr || 0);
                default: return new Date(b.data) - new Date(a.data);
            }
        });

        state.filteredData = processedData;

        renderKpiCards();
        renderAllCharts();
        renderTableAndPagination();
        updateTableColumns();
    };

    const renderKpiCards = () => {
        const data = state.filteredData;
        const totalInvestment = data.reduce((sum, item) => sum + parseFloat(item.investimento_total || 0), 0);
        const totalLeads = data.reduce((sum, item) => sum + parseInt(item.leads || 0), 0);
        const totalClicks = data.reduce((sum, item) => sum + parseInt(item.cliques_totais || 0), 0);
        const totalImpressions = data.reduce((sum, item) => sum + parseInt(item.impressoes || 0), 0);
        const cpl = totalLeads > 0 ? (totalInvestment / totalLeads) : 0;
        const ctr = totalImpressions > 0 ? (totalClicks / totalImpressions) * 100 : 0;

        document.getElementById('totalInvestmentCard').textContent = `R$ ${totalInvestment.toLocaleString('pt-BR', { minimumFractionDigits: 2, maximumFractionDigits: 2 })}`;
        document.getElementById('totalLeadsCard').textContent = totalLeads;
        document.getElementById('cplCard').textContent = `R$ ${cpl.toLocaleString('pt-BR', { minimumFractionDigits: 2, maximumFractionDigits: 2 })}`;
        document.getElementById('totalRevenueCard').textContent = totalClicks.toLocaleString('pt-BR');
        document.getElementById('roiCard').textContent = `${ctr.toFixed(2)}%`;
        document.getElementById('conversionRateCard').textContent = totalImpressions.toLocaleString('pt-BR');
    };

    const renderAllCharts = () => {
        drawComboChart('investmentChart', 'Performance de Tráfego por Dia', state.filteredData);
    };

    const drawComboChart = (elementId, title, data) => {
        const container = document.getElementById(elementId);
        if (!container || data.length === 0) {
            container.innerHTML = `<p style="text-align:center; padding-top: 50px; color: #999;">Dados insuficientes para o gráfico.</p>`;
            return;
        }
        const dataByDate = {};
        data.forEach(item => {
            const date = new Date(item.data + 'T00:00:00').toISOString();
            if (!dataByDate[date]) { dataByDate[date] = { investment: 0, leads: 0, clicks: 0 }; }
            dataByDate[date].investment += parseFloat(item.investimento_total || 0);
            dataByDate[date].leads += parseInt(item.leads || 0);
            dataByDate[date].clicks += parseInt(item.cliques_totais || 0);
        });
        const sortedDates = Object.keys(dataByDate).sort((a,b) => new Date(a) - new Date(b));
        
        const dataArray = [['Data', 'Investimento', 'Leads', 'Cliques']];
        sortedDates.forEach(dateStr => {
            const date = new Date(dateStr);
            const dataPoint = dataByDate[dateStr];
            dataArray.push([ date, dataPoint.investment, dataPoint.leads, dataPoint.clicks ]);
        });
        const dataTable = google.visualization.arrayToDataTable(dataArray);
        const options = { 
            title: title, vAxes: { 0: {title: 'Valor (R$)'}, 1: {title: 'Quantidade'} }, hAxis: {title: 'Data', format: 'dd/MM' }, 
            seriesType: 'bars', series: { 0: {type: 'line', targetAxisIndex: 0, color: '#ffc107'}, 1: {type: 'bars', targetAxisIndex: 1, color: '#0d6efd'}, 2: {type: 'bars', targetAxisIndex: 1, color: '#198754'} }, 
            chartArea: {'width': '85%', 'height': '70%'}, legend: {'position': 'top'} 
        };
        const chart = new google.visualization.ComboChart(container);
        chart.draw(dataTable, options);
    };
    
    const renderTableAndPagination = () => {
        const totalPages = Math.ceil(state.filteredData.length / rowsPerPage);
        currentPage = Math.min(currentPage, totalPages) || 1;
        const startIndex = (currentPage - 1) * rowsPerPage;
        const paginatedData = state.filteredData.slice(startIndex, startIndex + rowsPerPage);
        renderDataTable(paginatedData);
        renderPaginationControls(totalPages);
    };

    const renderDataTable = (data) => {
        const tableBody = document.querySelector('#leadsTable tbody');
        tableBody.innerHTML = data.length === 0 
            ? '<tr><td colspan="7" style="text-align:center;">Nenhum dado encontrado para os filtros aplicados.</td></tr>'
            : data.map(item => `
                <tr>
                    <td>${new Date(item.data + 'T00:00:00').toLocaleDateString('pt-BR')}</td>
                    <td>${item.alcance || 0}</td>
                    <td>${item.impressoes || 0}</td>
                    <td>${item.cliques_totais || 0}</td>
                    <td>${item.leads || 0}</td>
                    <td>${parseFloat(item.investimento_total || 0).toLocaleString('pt-BR', { style: 'currency', currency: 'BRL' })}</td>
                    <td>${(parseFloat(item.ctr || 0) * 100).toFixed(2)}%</td>
                </tr>
            `).join('');
            
        if (elements.highlightTopResults.checked && data.length > 0) {
            const rows = tableBody.querySelectorAll('tr');
            for(let i = 0; i < Math.min(3, rows.length); i++) {
                rows[i].classList.add('highlighted-row');
            }
        }
    };

    const renderPaginationControls = (totalPages) => {
        const controls = elements.paginationControls;
        controls.innerHTML = '';
        if (totalPages <= 1) return;

        const prevButton = document.createElement('button');
        prevButton.textContent = '« Anterior';
        prevButton.disabled = currentPage === 1;
        prevButton.addEventListener('click', () => {
            if (currentPage > 1) {
                currentPage--;
                renderTableAndPagination();
                updateTableColumns();
            }
        });
        controls.appendChild(prevButton);

        const pageIndicator = document.createElement('span');
        pageIndicator.textContent = `Página ${currentPage} de ${totalPages}`;
        controls.appendChild(pageIndicator);

        const nextButton = document.createElement('button');
        nextButton.textContent = 'Próximo »';
        nextButton.disabled = currentPage === totalPages;
        nextButton.addEventListener('click', () => {
            if (currentPage < totalPages) {
                currentPage++;
                renderTableAndPagination();
                updateTableColumns();
            }
        });
        controls.appendChild(nextButton);
    };
    
    if (await checkSession()) {
        await fetchAllData();
        applyFiltersAndRender();
        
        elements.applyFiltersButton.addEventListener('click', applyFiltersAndRender);
        
        elements.dateRangeSelect.addEventListener('change', async () => { 
            const isCustom = elements.dateRangeSelect.value === 'custom';
            elements.customDateFilters.style.display = isCustom ? 'flex' : 'none';
            if (!isCustom) {
                await fetchAllData();
                applyFiltersAndRender();
            }
        });

        elements.columnSelectorButton.addEventListener('click', (e) => {
            e.stopPropagation();
            elements.columnSelector.classList.toggle('visible');
        });
        
        elements.columnCheckboxes.forEach(cb => {
            cb.addEventListener('change', updateTableColumns);
        });

        document.addEventListener('click', () => {
            elements.columnSelector.classList.remove('visible');
        });

        elements.logoutButton.addEventListener('click', async () => { await fetch('api/auth.php?action=logout'); window.location.href = 'login.html'; });
        window.addEventListener('resize', renderAllCharts);
    }
}