// =========================================================================
// PAINEL CÓRTEX - SCRIPT DO PAINEL CENTRAL v2.0 (COM COMPARATIVO)
// =========================================================================

google.charts.load('current', {'packages':['corechart', 'bar', 'pie']});
google.charts.setOnLoadCallback(initializeDashboard);

// Estado global para armazenar dados atuais e anteriores
const state = {
    current: { leads: [], trafego: [], organico: [] },
    previous: { leads: [], trafego: [], organico: [] }
};

// Objeto para armazenar referências dos elementos do DOM
const elements = {
    doctorName: null,
    logoutButton: null,
    dateRangeSelect: null,
    customDateFilters: null,
    startDateInput: null,
    endDateInput: null,
    comparePeriod: null,
    statusFilter: null,
    channelFilter: null
};

/**
 * Inicializa o painel: busca elementos, define listeners e carrega dados iniciais.
 */
async function initializeDashboard() {
    // 1. Mapear todos os elementos do DOM
    elements.doctorName = document.getElementById('doctorName');
    elements.logoutButton = document.getElementById('logoutButton');
    elements.dateRangeSelect = document.getElementById('dateRange');
    elements.customDateFilters = document.getElementById('customDateFilters');
    elements.startDateInput = document.getElementById('startDate');
    elements.endDateInput = document.getElementById('endDate');
    elements.comparePeriod = document.getElementById('comparePeriod');
    elements.statusFilter = document.getElementById('statusFilter');
    elements.channelFilter = document.getElementById('channelFilter');

    // 2. Verificar sessão do usuário
    if (!(await checkSession())) {
        window.location.href = 'login.html';
        return;
    }

    // 3. Definir listeners de eventos para todos os filtros
    elements.logoutButton.addEventListener('click', logout);
    elements.dateRangeSelect.addEventListener('change', handleFilterChange);
    elements.startDateInput.addEventListener('change', handleFilterChange);
    elements.endDateInput.addEventListener('change', handleFilterChange);
    elements.comparePeriod.addEventListener('change', handleFilterChange);
    elements.statusFilter.addEventListener('change', render); // Filtros de leads apenas re-renderizam
    elements.channelFilter.addEventListener('change', render); // Filtros de leads apenas re-renderizam
    window.addEventListener('resize', renderAllCharts);

    // 4. Definir datas padrão para "Este Mês" (necessário para a 1ª carga)
    setDefaultDates();

    // 5. Carregar dados iniciais e popular filtros
    await fetchAllData(true); // true = é a carga inicial
    render();
}

/**
 * Verifica se a sessão do usuário está ativa.
 */
async function checkSession() {
    try {
        const response = await fetch('api/auth.php?action=check_session');
        if (!response.ok) throw new Error('Falha na rede');
        const result = await response.json();
        if (!result.loggedin) return false;
        
        elements.doctorName.textContent = `Olá, ${result.doctor_name}`;
        return true;
    } catch (e) {
        return false;
    }
}

/**
 * Efetua logout.
 */
async function logout() {
    await fetch('api/auth.php?action=logout');
    window.location.href = 'login.html';
}

/**
 * Define as datas padrão nos inputs com base na seleção do dateRange.
 */
function setDefaultDates() {
    const { startDate, endDate } = getDateRange(); // Pega datas (objeto Date)
    // Formata como 'YYYY-MM-DD' para os inputs type="date"
    elements.startDateInput.value = startDate.toISOString().split('T')[0];
    elements.endDateInput.value = endDate.toISOString().split('T')[0];
}

/**
 * Lida com qualquer mudança nos filtros principais (data ou comparativo).
 * Busca novos dados da API e re-renderiza o painel.
 */
async function handleFilterChange() {
    const range = elements.dateRangeSelect.value;
    elements.customDateFilters.style.display = range === 'custom' ? 'flex' : 'none';

    if (range !== 'custom') {
        setDefaultDates(); // Atualiza as datas customizadas se o range mudar
    }

    await fetchAllData(false); // false = não é a carga inicial
    render();
}

/**
 * Busca todos os dados da API (atuais e, se marcado, anteriores).
 */
async function fetchAllData(isInitialLoad = false) {
    try {
        const { startDate, endDate, range } = getDateRange(true); // Pega datas (string formatada)
        const compare = elements.comparePeriod.checked;
        
        const apiUrl = `api/data_handler_central.php?startDate=${startDate}&endDate=${endDate}&range=${range}&compare=${compare}`;
        
        const response = await fetch(apiUrl);
        if (!response.ok) throw new Error(`Falha ao buscar dados: ${response.statusText}`);
        
        const allData = await response.json();
        if (allData.error) throw new Error(allData.error);

        state.current = allData.current || { leads: [], trafego: [], organico: [] };
        state.previous = allData.previous || { leads: [], trafego: [], organico: [] };

        if (isInitialLoad) {
            populateFilters();
        }
    } catch (error) {
        console.error("Erro fatal ao buscar dados:", error);
        // Limpa o estado em caso de erro
        state.current = { leads: [], trafego: [], organico: [] };
        state.previous = { leads: [], trafego: [], organico: [] };
    }
}

/**
 * Popula os filtros de Status e Canal com base nos dados carregados.
 */
const populateFilters = () => {
    if (state.current.leads.length === 0) return;
    
    // Usando Set para pegar valores únicos e filtrando nulos/vazios
    const statuses = [...new Set(state.current.leads.map(lead => lead.status).filter(Boolean))];
    const channels = [...new Set(state.current.leads.map(lead => lead.canal_de_origem).filter(Boolean))];
    
    statuses.sort();
    channels.sort();

    elements.statusFilter.innerHTML = '<option value="all">Todos os Status</option>';
    statuses.forEach(status => {elements.statusFilter.innerHTML += `<option value="${status}">${status}</option>`});

    elements.channelFilter.innerHTML = '<option value="all">Todos os Canais</option>';
    channels.forEach(channel => {elements.channelFilter.innerHTML += `<option value="${channel}">${channel}</option>`});
};

/**
 * Calcula o intervalo de datas selecionado.
 * @param {boolean} forApi - Se true, formata as datas como string YYYY-MM-DD.
 * @returns {object} - { startDate, endDate, range }
 */
const getDateRange = (forApi = false) => {
    const range = elements.dateRangeSelect.value;
    const today = new Date();
    let startDate, endDate = new Date();
    
    today.setHours(0, 0, 0, 0);
    endDate.setHours(23, 59, 59, 999);

    switch (range) {
        case 'this_month': 
            startDate = new Date(today.getFullYear(), today.getMonth(), 1);
            endDate = new Date(today.getFullYear(), today.getMonth() + 1, 0, 23, 59, 59, 999); // Último dia do mês
            break;
        case 'last_30_days': 
            startDate = new Date();
            startDate.setDate(today.getDate() - 29);
            startDate.setHours(0, 0, 0, 0);
            break;
        case 'custom':
            startDate = elements.startDateInput.value ? new Date(elements.startDateInput.value + "T00:00:00") : new Date(today);
            endDate = elements.endDateInput.value ? new Date(elements.endDateInput.value + "T23:59:59") : new Date(today);
            break;
    }

    if (forApi) {
        const formatDate = (date) => date.toISOString().split('T')[0];
        return { startDate: formatDate(startDate), endDate: formatDate(endDate), range: range };
    }
    
    return { startDate, endDate, range: range };
};

/**
 * Função principal de renderização. Chama os cálculos de KPI e Gráficos.
 */
const render = () => {
    renderKpiCards();
    renderAllCharts();
};

/**
 * Renderiza os 5 KPIs, incluindo a lógica de comparação.
 */
const renderKpiCards = () => {
    // 1. Calcular métricas do período ATUAL
    const currentInvestimento = state.current.trafego.reduce((sum, item) => sum + parseFloat(item.investimento_total || 0), 0);
    const currentLeads = state.current.leads.length; // Total de leads da tabela leads
    const currentCpl = currentLeads > 0 ? (currentInvestimento / currentLeads) : 0;
    const currentSeguidores = state.current.organico.reduce((sum, item) => sum + parseInt(item.novos_seguidores || 0), 0);
    const currentReceita = state.current.leads.reduce((sum, item) => sum + parseFloat(item.preco || 0), 0);
    const currentRoas = currentInvestimento > 0 ? (currentReceita / currentInvestimento) : 0;

    // 2. Calcular métricas do período ANTERIOR
    const prevInvestimento = state.previous.trafego.reduce((sum, item) => sum + parseFloat(item.investimento_total || 0), 0);
    const prevLeads = state.previous.leads.length;
    const prevCpl = prevLeads > 0 ? (prevInvestimento / prevLeads) : 0;
    const prevSeguidores = state.previous.organico.reduce((sum, item) => sum + parseInt(item.novos_seguidores || 0), 0);
    const prevReceita = state.previous.leads.reduce((sum, item) => sum + parseFloat(item.preco || 0), 0);
    const prevRoas = prevInvestimento > 0 ? (prevReceita / prevInvestimento) : 0;

    // 3. Renderizar HTML (com helpers de formatação)
    const formatBRL = (value) => value.toLocaleString('pt-BR', { minimumFractionDigits: 2, maximumFractionDigits: 2 });
    const formatNum = (value) => value.toLocaleString('pt-BR');
    const formatROAS = (value) => `${value.toFixed(2)}x`;

    document.getElementById('kpiInvestimento').innerHTML = `R$ ${formatBRL(currentInvestimento)} ${getComparisonString(currentInvestimento, prevInvestimento, 'BRL')}`;
    document.getElementById('kpiLeads').innerHTML = `${formatNum(currentLeads)} ${getComparisonString(currentLeads, prevLeads, 'num')}`;
    document.getElementById('kpiCpl').innerHTML = `R$ ${formatBRL(currentCpl)} ${getComparisonString(currentCpl, prevCpl, 'BRL_invert')}`; // CPL é melhor quando MENOR
    document.getElementById('kpiSeguidores').innerHTML = `${formatNum(currentSeguidores)} ${getComparisonString(currentSeguidores, prevSeguidores, 'num')}`;
    document.getElementById('kpiRoas').innerHTML = `${formatROAS(currentRoas)} ${getComparisonString(currentRoas, prevRoas, 'num')}`;
};

/**
 * Cria a string de comparação (ex: "+10.5%")
 * @param {number} current - Valor atual
 * @param {number} previous - Valor anterior
 * @param {string} type - 'num', 'BRL', ou 'BRL_invert' (para CPL)
 */
function getComparisonString(current, previous, type = 'num') {
    if (!elements.comparePeriod.checked || previous === 0) {
        return (current > 0 && previous === 0) ? '<span class="compare-text compare-up">(Novo)</span>' : '';
    }

    const diff = current - previous;
    const percentChange = (diff / previous) * 100;
    
    let arrow = 'neutral';
    if (percentChange > 0) arrow = 'up';
    if (percentChange < 0) arrow = 'down';

    // Inverte a lógica de cor para CPL (menor é melhor)
    if (type === 'BRL_invert') {
        if (arrow === 'up') arrow = 'down-bad'; // temporário
        if (arrow === 'down') arrow = 'up';
        if (arrow === 'down-bad') arrow = 'down';
    }

    const symbol = percentChange > 0 ? '+' : '';
    
    return `<span class="compare-text compare-${arrow}">(${symbol}${percentChange.toFixed(1)}%)</span>`;
}


/**
 * Chama todas as funções de desenho de gráfico.
 */
const renderAllCharts = () => {
    drawGeralPerformanceChart();
    drawLeadFunnelCharts(); // Função única para os dois gráficos de leads
};

/**
 * GRÁFICO 1: Performance Geral (Alcance, Cliques, Leads, Seguidores)
 * Conforme solicitado pelo padrão.
 */
const drawGeralPerformanceChart = () => {
    const dataByDate = {};

    // Helper para inicializar um dia
    const initDate = (date) => {
        if (!dataByDate[date]) {
            dataByDate[date] = { alcance: 0, cliques: 0, leads: 0, seguidores: 0 };
        }
    };

    // 1. Processar dados de Tráfego (cliques, alcance)
    state.current.trafego.forEach(item => {
        const date = item.data;
        initDate(date);
        dataByDate[date].cliques += parseInt(item.cliques_totais || 0);
        dataByDate[date].alcance += parseInt(item.alcance || 0);
    });

    // 2. Processar dados Orgânicos (seguidores, alcance)
    state.current.organico.forEach(item => {
        const date = item.data;
        initDate(date);
        dataByDate[date].seguidores += parseInt(item.novos_seguidores || 0);
        dataByDate[date].alcance += parseInt(item.alcance || 0);
    });

    // 3. Processar dados de Leads (contagem de leads)
    state.current.leads.forEach(item => {
        // A data do lead é 'data_insercao'
        const date = item.data_insercao; 
        initDate(date);
        dataByDate[date].leads += 1;
    });

    // 4. Montar o array do gráfico
    const sortedDates = Object.keys(dataByDate).sort((a,b) => new Date(a) - new Date(b));
    const dataArray = [['Data', 'Alcance', 'Cliques', 'Leads', 'Novos Seguidores']];
    sortedDates.forEach(date => {
        const entry = dataByDate[date];
        dataArray.push([new Date(date + 'T00:00:00'), entry.alcance, entry.cliques, entry.leads, entry.seguidores]);
    });

    const chartEl = document.getElementById('geralPerformanceChart');
    if (dataArray.length <= 1) {
        chartEl.innerHTML = `<p class="chart-no-data">Dados insuficientes para o gráfico de Performance Geral.</p>`;
        return;
    }
    
    const dataTable = google.visualization.arrayToDataTable(dataArray);
    const options = { 
        title: 'Performance Geral por Dia (Tráfego + Orgânico)', 
        vAxis: {title: 'Quantidade'}, 
        hAxis: {
            title: 'Data', 
            format: 'dd/MM/yy' // FORMATO DE DATA ATUALIZADO
        },
        seriesType: 'bars', 
        series: { 
            0: {type: 'line', color: '#6c757d'} // Alcance como linha
        },
        legend: { position: 'top' },
        chartArea: {'width': '85%', 'height': '70%'}
    };
    const chart = new google.visualization.ComboChart(chartEl);
    chart.draw(dataTable, options);
};

/**
 * Renderiza os DOIS gráficos de leads (Status e Origem), pois ambos usam os mesmos filtros.
 */
const drawLeadFunnelCharts = () => {
    // Aplicar filtros de Status e Canal APENAS para estes gráficos
    const statusFiltro = elements.statusFilter.value;
    const channelFiltro = elements.channelFilter.value;

    const filteredLeads = state.current.leads.filter(item => {
        const statusMatch = statusFiltro === 'all' || item.status === statusFiltro;
        const channelMatch = channelFiltro === 'all' || item.canal_de_origem === channelFiltro;
        return statusMatch && channelMatch;
    });

    // Chamar as funções de desenho específicas
    drawFunilStatusChart(filteredLeads, state.current.leads.length);
    drawOrigemLeadsChart(filteredLeads);
};

/**
 * GRÁFICO 2: Funil de Status dos Leads (com Taxa de Conversão)
 * @param {Array} filteredLeads - Leads já filtrados por status/canal
 * @param {number} totalLeadsPeriodo - Total de leads ANTES dos filtros de status/canal
 */
const drawFunilStatusChart = (filteredLeads, totalLeadsPeriodo) => {
    const groupedData = filteredLeads.reduce((acc, item) => {
        const key = item.status || 'Não definido';
        acc[key] = (acc[key] || 0) + 1;
        return acc;
    }, {});

    // Calcular taxa de conversão de Agendados (sobre o total do período, não o filtrado)
    const agendados = state.current.leads.reduce((sum, item) => (item.status === 'Agendado' ? sum + 1 : sum), 0);
    const conversaoAgendado = totalLeadsPeriodo > 0 ? (agendados / totalLeadsPeriodo) * 100 : 0;
    const newTitle = `Funil de Status (Conversão Agendados: ${conversaoAgendado.toFixed(1)}%)`;

    // Montar array do gráfico
    const dataArray = [['Status', 'Quantidade']];
    // Ordenar por quantidade, decrescente
    const sortedData = Object.entries(groupedData).sort(([,a],[,b]) => b-a);
    sortedData.forEach(([key, value]) => {
        dataArray.push([key, value]);
    });

    const chartEl = document.getElementById('funilStatusChart');
    if (dataArray.length <= 1) {
        chartEl.innerHTML = `<p class="chart-no-data">Dados insuficientes para o Funil de Status.</p>`;
        return;
    }

    const dataTable = google.visualization.arrayToDataTable(dataArray);
    const options = { 
        title: newTitle, // TÍTULO ATUALIZADO
        legend: { position: "none" },
        chartArea: {'width': '75%', 'height': '70%'}
    };
    const chart = new google.visualization.BarChart(chartEl);
    chart.draw(dataTable, options);
};

/**
 * GRÁFICO 3: Origem dos Leads (Pizza)
 * @param {Array} filteredLeads - Leads já filtrados por status/canal
 */
const drawOrigemLeadsChart = (filteredLeads) => {
    const groupedData = filteredLeads.reduce((acc, item) => {
        const key = item.canal_de_origem || 'Não definido';
        acc[key] = (acc[key] || 0) + 1;
        return acc;
    }, {});
    
    const dataArray = [['Canal', 'Quantidade']];
    Object.entries(groupedData).forEach(([key, value]) => {
        dataArray.push([key, value]);
    });

    const chartEl = document.getElementById('origemLeadsChart');
    if (dataArray.length <= 1) {
        chartEl.innerHTML = `<p class="chart-no-data">Dados insuficientes para o gráfico de Origem.</p>`;
        return;
    }
    
    const dataTable = google.visualization.arrayToDataTable(dataArray);
    const options = { 
        title: 'Origem dos Leads (Filtrado)', 
        pieHole: 0.4 
    };
    const chart = new google.visualization.PieChart(chartEl);
    chart.draw(dataTable, options);
};