// =========================================================================
// PAINEL CÓRTEX - SCRIPT DE DESEMPENHO ORGÂNICO v11.0 (COM COMPARAÇÃO)
// =========================================================================

google.charts.load('current', {'packages':['corechart', 'bar']});
google.charts.setOnLoadCallback(run);

// MODIFICADO: O estado agora suporta dois períodos (A e B)
const state = {
    periodA: { allData: [], filteredData: [] },
    periodB: { allData: [], filteredData: [] },
    isComparing: false,
    combinedFilteredData: [] // Para a tabela no modo de comparação
};

let currentPage = 1;
const rowsPerPage = 15;

function updateTableColumns() {
    // ... (Esta função continua igual)
    const checkboxes = document.querySelectorAll('#columnSelector .dropdown-options input[type="checkbox"]');
    checkboxes.forEach(checkbox => {
        const columnIndex = parseInt(checkbox.getAttribute('data-column')) + 1;
        const cells = document.querySelectorAll(`#dataTable th:nth-child(${columnIndex}), #dataTable td:nth-child(${columnIndex})`);
        if (cells.length > 0) {
            cells.forEach(cell => {
                cell.style.display = checkbox.checked ? '' : 'none';
            });
        }
    });
}

async function run() {
    const elements = {
        doctorName: document.getElementById('doctorName'),
        logoutButton: document.getElementById('logoutButton'),
        dateRangeSelect: document.getElementById('dateRange'),
        customDateFilters: document.getElementById('customDateFilters'),
        startDateInput: document.getElementById('startDate'),
        endDateInput: document.getElementById('endDate'),
        
        // NOVO: Elementos para o filtro de comparação
        compareDateFilters: document.getElementById('compareDateFilters'),
        compareStartDateA: document.getElementById('compareStartDateA'),
        compareEndDateA: document.getElementById('compareEndDateA'),
        compareStartDateB: document.getElementById('compareStartDateB'),
        compareEndDateB: document.getElementById('compareEndDateB'),

        applyFiltersButton: document.getElementById('applyFiltersButton'),
        paginationControls: document.getElementById('paginationControls'),
        orderByFilter: document.getElementById('orderByFilter'),
        minFollowersFilter: document.getElementById('minFollowersFilter'),
        minEngagementFilter: document.getElementById('minEngagementFilter'),
        highlightTopResults: document.getElementById('highlightTopResults'),
        columnSelector: document.getElementById('columnSelector'),
        columnSelectorButton: document.querySelector('#columnSelector .dropdown-button'),
        columnCheckboxes: document.querySelectorAll('#columnSelector .dropdown-options input')
    };

    const checkSession = async () => {
        // ... (Esta função continua igual)
        try {
            const response = await fetch('api/auth.php?action=check_session');
            if (!response.ok) throw new Error('Falha na rede');
            const result = await response.json();
            if (!result.loggedin) { window.location.href = 'login.html'; return false; }
            elements.doctorName.textContent = `Olá, ${result.doctor_name}`;
            return true;
        } catch (e) { window.location.href = 'login.html'; return false; }
    };

    // MODIFICADO: Lógica de busca de dados foi centralizada aqui
    const fetchAllData = async () => {
        const range = elements.dateRangeSelect.value;
        state.isComparing = (range === 'compare');

        // Função auxiliar para buscar dados da API
        const fetchDataForApi = async (startDate, endDate) => {
            try {
                // Se as datas não existirem (ex: "Todo o Período"), passa strings vazias
                const sDate = startDate || '';
                const eDate = endDate || '';
                const apiUrl = `api/data_handler_organico.php?startDate=${sDate}&endDate=${eDate}`;
                const response = await fetch(apiUrl);
                if (!response.ok) throw new Error('Falha ao buscar dados da API Orgânico');
                const data = await response.json();
                if(data.error) throw new Error(data.error);
                return data;
            } catch (error) {
                console.error("Erro ao buscar dados orgânicos:", error);
                return []; // Retorna vazio em caso de erro
            }
        };

        if (state.isComparing) {
            // === MODO DE COMPARAÇÃO ===
            const sA = elements.compareStartDateA.value;
            const eA = elements.compareEndDateA.value;
            const sB = elements.compareStartDateB.value;
            const eB = elements.compareEndDateB.value;

            if (!sA || !eA || !sB || !eB) {
                alert("Por favor, preencha ambos os períodos (A e B) para comparar.");
                state.periodA.allData = [];
                state.periodB.allData = [];
                return; // Interrompe a execução
            }

            // Busca os dois períodos em paralelo
            const [dataA, dataB] = await Promise.all([
                fetchDataForApi(sA, eA),
                fetchDataForApi(sB, eB)
            ]);
            state.periodA.allData = dataA;
            state.periodB.allData = dataB;

        } else {
            // === MODO PADRÃO ===
            const { startDate, endDate } = getDateRange(true); // Usa a função existente
            state.periodA.allData = await fetchDataForApi(startDate, endDate);
            state.periodB.allData = []; // Limpa os dados do período B
        }
    };
    
    // MODIFICADO: Esta função agora aplica filtros aos dados já buscados
    const applyFiltersAndRender = () => {
        const minFollowers = parseInt(elements.minFollowersFilter.value) || 0;
        const minEngagement = parseInt(elements.minEngagementFilter.value) || 0;
        const orderBy = elements.orderByFilter.value;

        // Função auxiliar para filtrar e ordenar um conjunto de dados
        const processData = (data) => {
            let processed = data.filter(item => {
                const followers = parseInt(item.novos_seguidores || 0);
                const engagement = parseInt(item.engajamento || 0);
                return followers >= minFollowers && engagement >= minEngagement;
            });

            processed.sort((a, b) => {
                switch (orderBy) {
                    case 'novos_seguidores_desc': return parseInt(b.novos_seguidores || 0) - parseInt(a.novos_seguidores || 0);
                    case 'engajamento_desc': return parseInt(b.engajamento || 0) - parseInt(a.engajamento || 0);
                    case 'alcance_desc': return parseInt(b.alcance || 0) - parseInt(a.alcance || 0);
                    case 'visitas_no_perfil_desc': return parseInt(b.visitas_no_perfil || 0) - parseInt(a.visitas_no_perfil || 0);
                    case 'cliques_no_link_desc': return parseInt(b.cliques_no_link || 0) - parseInt(a.cliques_no_link || 0);
                    default: return new Date(b.data) - new Date(a.data);
                }
            });
            return processed;
        };

        // Processa o Período A
        state.periodA.filteredData = processData(state.periodA.allData);

        if (state.isComparing) {
            // Processa o Período B
            state.periodB.filteredData = processData(state.periodB.allData);
        } else {
            state.periodB.filteredData = [];
        }

        // Para a tabela: combina os dados dos dois períodos e re-ordena
        state.combinedFilteredData = [...state.periodA.filteredData, ...state.periodB.filteredData];
        state.combinedFilteredData.sort((a, b) => {
             switch (orderBy) {
                case 'novos_seguidores_desc': return parseInt(b.novos_seguidores || 0) - parseInt(a.novos_seguidores || 0);
                case 'engajamento_desc': return parseInt(b.engajamento || 0) - parseInt(a.engajamento || 0);
                case 'alcance_desc': return parseInt(b.alcance || 0) - parseInt(a.alcance || 0);
                case 'visitas_no_perfil_desc': return parseInt(b.visitas_no_perfil || 0) - parseInt(a.visitas_no_perfil || 0);
                case 'cliques_no_link_desc': return parseInt(b.cliques_no_link || 0) - parseInt(a.cliques_no_link || 0);
                default: return new Date(b.data) - new Date(a.data);
            }
        });

        // Chama as funções de renderização
        renderKpiCards();
        renderAllCharts();
        renderTableAndPagination();
        updateTableColumns();
    };
    
    const getDateRange = (forApi = false) => {
        // ... (Esta função continua igual)
        const range = elements.dateRangeSelect.value;
        const today = new Date();
        let startDate, endDate = new Date();
        today.setHours(0, 0, 0, 0);
        endDate.setHours(23, 59, 59, 999);
        switch (range) {
            case 'today': startDate = new Date(today); break;
            case 'this_week': startDate = new Date(today); startDate.setDate(startDate.getDate() - today.getDay()); break;
            case 'this_month': startDate = new Date(today.getFullYear(), today.getMonth(), 1); break;
            case 'last_30_days': startDate = new Date(); startDate.setDate(today.getDate() - 29); startDate.setHours(0,0,0,0); break;
            case 'custom':
                startDate = elements.startDateInput.value ? new Date(elements.startDateInput.value + "T00:00:00") : null;
                endDate = elements.endDateInput.value ? new Date(elements.endDateInput.value + "T23:59:59") : null;
                break;
            // NOVO: 'compare' não retorna datas aqui, pois é tratado em fetchAllData
            case 'compare':
                startDate = null;
                endDate = null;
                break;
            default: startDate = null; endDate = null; break; // 'all'
        }
        if (forApi) {
            const formatDate = (date) => date ? date.toISOString().split('T')[0] : '';
            if (range === 'all' || range === 'compare') { return { startDate: '', endDate: '' }; }
            return { startDate: formatDate(startDate), endDate: formatDate(endDate) };
        }
        return { startDate, endDate };
    };

    // MODIFICADO: Atualizado para lidar com os novos spans (value-a, value-b)
    const renderKpiCards = () => {
        // Função auxiliar para calcular métricas de um conjunto de dados
        const calculateMetrics = (data) => {
            const novosSeguidores = data.reduce((sum, item) => sum + parseInt(item.novos_seguidores || 0), 0);
            const engajamento = data.reduce((sum, item) => sum + parseInt(item.engajamento || 0), 0);
            const alcance = data.reduce((sum, item) => sum + parseInt(item.alcance || 0), 0);
            const visitas = data.reduce((sum, item) => sum + parseInt(item.visitas_no_perfil || 0), 0);
            const cliquesLink = data.reduce((sum, item) => sum + parseInt(item.cliques_no_link || 0), 0);
            const totalTaxaEngajamento = data.reduce((sum, item) => sum + parseFloat(item.taxa_engajamento || 0), 0);
            const taxaEngajamentoMedia = data.length > 0 ? (totalTaxaEngajamento / data.length) * 100 : 0;

            return {
                novosSeguidores: novosSeguidores.toLocaleString('pt-BR'),
                engajamento: engajamento.toLocaleString('pt-BR'),
                alcance: alcance.toLocaleString('pt-BR'),
                visitas: visitas.toLocaleString('pt-BR'),
                cliquesLink: cliquesLink.toLocaleString('pt-BR'),
                taxaEngajamentoMedia: `${taxaEngajamentoMedia.toFixed(2)}%`
            };
        };

        // Calcula métricas para o Período A
        const metricsA = calculateMetrics(state.periodA.filteredData);

        // Função auxiliar para atualizar um card
        const updateCard = (cardId, valueA, valueB = '') => {
            const card = document.getElementById(cardId);
            if (!card) return;
            
            const elA = card.querySelector('.value-a');
            const elB = card.querySelector('.value-b');
            const elSep = card.querySelector('.compare-separator');

            if (elA) elA.textContent = valueA;

            if (state.isComparing) {
                if (elB) elB.textContent = valueB;
                if (elB) elB.style.display = '';
                if (elSep) elSep.style.display = '';
            } else {
                if (elB) elB.style.display = 'none';
                if (elSep) elSep.style.display = 'none';
            }
        };

        if (state.isComparing) {
            const metricsB = calculateMetrics(state.periodB.filteredData);
            updateCard('newFollowersCard', metricsA.novosSeguidores, metricsB.novosSeguidores);
            updateCard('totalEngagementCard', metricsA.engajamento, metricsB.engajamento);
            updateCard('totalReachCard', metricsA.alcance, metricsB.alcance);
            updateCard('profileVisitsCard', metricsA.visitas, metricsB.visitas);
            updateCard('linkClicksCard', metricsA.cliquesLink, metricsB.cliquesLink);
            updateCard('avgEngagementRateCard', metricsA.taxaEngajamentoMedia, metricsB.taxaEngajamentoMedia);
        } else {
            // Modo padrão: apenas atualiza o valor A e garante que B esteja oculto
            updateCard('newFollowersCard', metricsA.novosSeguidores);
            updateCard('totalEngagementCard', metricsA.engajamento);
            updateCard('totalReachCard', metricsA.alcance);
            updateCard('profileVisitsCard', metricsA.visitas);
            updateCard('linkClicksCard', metricsA.cliquesLink);
            updateCard('avgEngagementRateCard', metricsA.taxaEngajamentoMedia);
        }
    };

    // MODIFICADO: Passa ambos os conjuntos de dados para o gráfico
    const renderAllCharts = () => {
        drawMainChart(
            'mainChart', 
            'Performance Orgânica por Dia', 
            state.periodA.filteredData, 
            state.isComparing ? state.periodB.filteredData : null // Passa B apenas se estiver comparando
        );
    };

    // MODIFICADO: Função do gráfico corrigida para tratar 'null' vs '0'
        const drawMainChart = (elementId, title, dataA, dataB) => {
        const container = document.getElementById(elementId);
        const hasDataA = dataA && dataA.length > 0;
        const hasDataB = dataB && dataB.length > 0;

        if (!container || (!hasDataA && !hasDataB)) {
            container.innerHTML = `<p style="text-align:center; padding-top: 50px; color: #999;">Dados insuficientes para o gráfico.</p>`;
            return;
        }

        const dataByDate = {};
        
        // Processa Período A
        if(hasDataA) {
            dataA.forEach(item => {
                const dateKey = item.data;
                if (!dataByDate[dateKey]) dataByDate[dateKey] = {};
                dataByDate[dateKey].a_alcance = (dataByDate[dateKey].a_alcance || 0) + parseInt(item.alcance || 0);
                dataByDate[dateKey].a_seguidores = (dataByDate[dateKey].a_seguidores || 0) + parseInt(item.novos_seguidores || 0);
                dataByDate[dateKey].a_engajamento = (dataByDate[dateKey].a_engajamento || 0) + parseInt(item.engajamento || 0);
            });
        }
    
        // Processa Período B (se existir)
        if (hasDataB) {
            dataB.forEach(item => {
                const dateKey = item.data;
                if (!dataByDate[dateKey]) dataByDate[dateKey] = {};
                dataByDate[dateKey].b_alcance = (dataByDate[dateKey].b_alcance || 0) + parseInt(item.alcance || 0);
                dataByDate[dateKey].b_seguidores = (dataByDate[dateKey].b_seguidores || 0) + parseInt(item.novos_seguidores || 0);
                dataByDate[dateKey].b_engajamento = (dataByDate[dateKey].b_engajamento || 0) + parseInt(item.engajamento || 0);
            });
        }
    
        // Pega todas as datas de A e B e ordena
        const sortedDates = Object.keys(dataByDate).sort((a, b) => new Date(a) - new Date(b));
    
        const dataArray = [];
        let seriesOptions = {};

        if (hasDataB) {
            // MODO COMPARAÇÃO: 6 colunas de dados
            dataArray.push(['Data', 'Alcance (A)', 'Seguidores (A)', 'Engajamento (A)', 'Alcance (B)', 'Seguidores (B)', 'Engajamento (B)']);
            sortedDates.forEach(dateKey => {
                const date = new Date(dateKey + 'T00:00:00');
                const dataPoint = dataByDate[dateKey];
                
                // ==========================================================
                // AQUI ESTÁ A CORREÇÃO: Trocamos '|| null' por '|| 0'
                // ==========================================================
                dataArray.push([
                    date,
                    dataPoint.a_alcance || 0,
                    dataPoint.a_seguidores || 0,
                    dataPoint.a_engajamento || 0,
                    dataPoint.b_alcance || 0,
                    dataPoint.b_seguidores || 0,
                    dataPoint.b_engajamento || 0
                ]);
            });
            
            // Configurações das séries para comparação
            seriesOptions = {
                0: { type: 'line', color: '#0d6efd', lineDashStyle: [4, 4] }, // Alcance A (tracejado)
                1: { type: 'bars', color: '#198754' }, // Seguidores A
                2: { type: 'bars', color: '#ffc107' }, // Engajamento A
                3: { type: 'line', color: '#0dcaf0' }, // Alcance B (linha sólida)
                4: { type: 'bars', color: '#084229' }, // Seguidores B (cor mais escura)
                5: { type: 'bars', color: '#c69500' }  // Engajamento B (cor mais escura)
            };

        } else {
            // MODO PADRÃO: 3 colunas de dados
            dataArray.push(['Data', 'Alcance', 'Novos Seguidores', 'Engajamento']);
            sortedDates.forEach(dateKey => {
                const date = new Date(dateKey + 'T00:00:00');
                const dataPoint = dataByDate[dateKey];
                // No modo padrão, '|| null' pode ser '|| 0' também, por consistência
                dataArray.push([ 
                    date, 
                    dataPoint.a_alcance || 0, 
                    dataPoint.a_seguidores || 0, 
                    dataPoint.a_engajamento || 0
                ]);
            });
            
            // Configurações das séries padrão
            seriesOptions = {
                0: { type: 'line', color: '#0d6efd' }, 
                1: { type: 'bars', color: '#198754' }, 
                2: { type: 'bars', color: '#ffc107' }
            };
        }

        const dataTable = google.visualization.arrayToDataTable(dataArray);
        const options = { 
            title: title, 
            vAxis: { title: 'Quantidade' }, 
            hAxis: { title: 'Data', format: 'dd/MM' }, 
            seriesType: 'bars', 
            series: seriesOptions, 
            legend: { 'position': 'top' },
            // NOVO: Garante que valores 0 sejam tratados corretamente
            bar: { groupWidth: '75%' }, 
            isStacked: false 
        };
    
        const chart = new google.visualization.ComboChart(container);
        chart.draw(dataTable, options);
    };

    // MODIFICADO: Agora usa 'state.combinedFilteredData'
    const renderTableAndPagination = () => {
        const totalPages = Math.ceil(state.combinedFilteredData.length / rowsPerPage); // Usa os dados combinados
        currentPage = Math.min(currentPage, totalPages) || 1;
        const startIndex = (currentPage - 1) * rowsPerPage;
        const paginatedData = state.combinedFilteredData.slice(startIndex, startIndex + rowsPerPage); // Usa os dados combinados
        
        renderDataTable(paginatedData);
        renderPaginationControls(totalPages);
    };

    const renderDataTable = (data) => {
        // ... (Esta função continua igual, ela só renderiza o que recebe)
        const tableBody = document.querySelector('#dataTable tbody');
        tableBody.innerHTML = data.length === 0 
            ? '<tr><td colspan="9">Nenhum dado encontrado para o período.</td></tr>'
            : data.map(item => `
                <tr>
                    <td>${new Date(item.data + 'T00:00:00').toLocaleDateString('pt-BR')}</td>
                    <td>${parseInt(item.novos_seguidores || 0).toLocaleString('pt-BR')}</td>
                    <td>${parseInt(item.engajamento || 0).toLocaleString('pt-BR')}</td>
                    <td>${parseInt(item.alcance || 0).toLocaleString('pt-BR')}</td>
                    <td>${parseInt(item.visitas_no_perfil || 0).toLocaleString('pt-BR')}</td>
                    <td>${parseInt(item.cliques_no_link || 0).toLocaleString('pt-BR')}</td>
                    <td>${parseInt(item.likes || 0).toLocaleString('pt-BR')}</td>
                    <td>${parseInt(item.comentarios || 0).toLocaleString('pt-BR')}</td>
                    <td>${parseInt(item.salvamentos || 0).toLocaleString('pt-BR')}</td>
                </tr>
            `).join('');
        
        if (elements.highlightTopResults.checked && data.length > 0) {
            const rows = tableBody.querySelectorAll('tr');
            for(let i = 0; i < Math.min(3, rows.length); i++) {
                rows[i].classList.add('highlighted-row');
            }
        }
    };

    const renderPaginationControls = (totalPages) => {
        // ... (Esta função continua igual)
        elements.paginationControls.innerHTML = '';
        if (totalPages <= 1) return;
        const prevButton = document.createElement('button');
        prevButton.textContent = '« Anterior';
        prevButton.disabled = currentPage === 1;
        prevButton.addEventListener('click', () => {
            if (currentPage > 1) { currentPage--; renderTableAndPagination(); updateTableColumns(); }
        });
        elements.paginationControls.appendChild(prevButton);
        const pageIndicator = document.createElement('span');
        pageIndicator.textContent = `Página ${currentPage} de ${totalPages > 0 ? totalPages : 1}`;
        elements.paginationControls.appendChild(pageIndicator);
        const nextButton = document.createElement('button');
        nextButton.textContent = 'Próximo »';
        nextButton.disabled = currentPage === totalPages || totalPages === 0;
        nextButton.addEventListener('click', () => {
            if (currentPage < totalPages) { currentPage++; renderTableAndPagination(); updateTableColumns(); }
        });
        elements.paginationControls.appendChild(nextButton);
    };

    // MODIFICADO: Lógica de inicialização e listeners
    if (await checkSession()) {
        // Carga inicial
        await fetchAllData();
        applyFiltersAndRender();
        
        // Listener principal para o botão "Aplicar Filtros"
        // Agora ele busca os dados E renderiza
        elements.applyFiltersButton.addEventListener('click', async () => {
            await fetchAllData();
            applyFiltersAndRender();
        });

        // Outros filtros (ordenar, min, etc.) apenas re-filtram, não buscam na API
        elements.orderByFilter.addEventListener('change', applyFiltersAndRender);
        elements.minFollowersFilter.addEventListener('input', applyFiltersAndRender);
        elements.minEngagementFilter.addEventListener('input', applyFiltersAndRender);
        elements.highlightTopResults.addEventListener('change', applyFiltersAndRender);

        // Listener para o seletor de data
        elements.dateRangeSelect.addEventListener('change', async () => { 
            const range = elements.dateRangeSelect.value;
            // Apenas controla a visibilidade dos campos
            elements.customDateFilters.style.display = (range === 'custom') ? 'flex' : 'none';
            elements.compareDateFilters.style.display = (range === 'compare') ? 'flex' : 'none';

            // Se for um range pré-definido, busca e renderiza imediatamente
            if (range !== 'custom' && range !== 'compare') {
                await fetchAllData();
                applyFiltersAndRender();
            }
        });

        // Listener do seletor de colunas
        if (elements.columnSelectorButton) {
            elements.columnSelectorButton.addEventListener('click', (e) => {
                e.stopPropagation();
                elements.columnSelector.classList.toggle('visible');
            });
            elements.columnCheckboxes.forEach(cb => {
                cb.addEventListener('change', updateTableColumns);
            });
            document.addEventListener('click', (e) => {
                if(elements.columnSelector && !elements.columnSelector.contains(e.target)) {
                    elements.columnSelector.classList.remove('visible');
                }
            });
        }

        // Listener de logout e resize
        elements.logoutButton.addEventListener('click', async () => { await fetch('api/auth.php?action=logout'); window.location.href = 'login.html'; });
        window.addEventListener('resize', renderAllCharts);
    }
}